import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Clock, CheckCircle, MessageSquare, Star, Package, Tent, Bed, Backpack, TrendingUp, Shield, Zap, BarChart3, Mountain, MapPin, Camera, ArrowRight, Quote, User } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const navigate = useNavigate()
  const [rentals, setRentals] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('campingRentals')
    const defaultRentals = [
      {
        id: 1,
        item: 'Палатка 4-местная',
        category: 'tents',
        startDate: '2025-05-15',
        returnDate: '2025-05-20',
        status: 'active',
        review: null
      },
      {
        id: 2,
        item: 'Спальник -10°C',
        category: 'sleeping',
        startDate: '2025-04-20',
        returnDate: '2025-04-25',
        status: 'completed',
        review: {
          rating: 5,
          text: 'Отличное качество! Очень теплый и удобный спальник.'
        }
      },
      {
        id: 3,
        item: 'Рюкзак 60л',
        category: 'backpacks',
        startDate: '2025-03-10',
        returnDate: '2025-03-15',
        status: 'completed',
        review: null
      }
    ]
    
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        // Проверяем, что данные валидны и имеют даты
        const validRentals = parsed.filter(rental => 
          rental.startDate && rental.returnDate && 
          rental.startDate.match(/^\d{4}-\d{2}-\d{2}$/) &&
          rental.returnDate.match(/^\d{4}-\d{2}-\d{2}$/)
        )
        if (validRentals.length > 0) {
          setRentals(parsed)
        } else {
          // Если данные невалидны, используем дефолтные
          setRentals(defaultRentals)
          localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
        }
      } catch (e) {
        setRentals(defaultRentals)
        localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
      }
    } else {
      setRentals(defaultRentals)
      localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
    }
  }, [])

  const handleReviewSubmit = (rentalId, rating, text) => {
    const updated = rentals.map(rental => {
      if (rental.id === rentalId) {
        return {
          ...rental,
          review: { rating, text }
        }
      }
      return rental
    })
    setRentals(updated)
    localStorage.setItem('campingRentals', JSON.stringify(updated))
  }

  const handleComplete = (rentalId) => {
    const updated = rentals.map(rental => {
      if (rental.id === rentalId) {
        return {
          ...rental,
          status: 'completed'
        }
      }
      return rental
    })
    setRentals(updated)
    localStorage.setItem('campingRentals', JSON.stringify(updated))
  }

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      // Правильно парсим дату в формате YYYY-MM-DD
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  const getCategoryIcon = (category) => {
    switch(category) {
      case 'tents':
        return Tent
      case 'sleeping':
        return Bed
      case 'backpacks':
        return Backpack
      default:
        return Package
    }
  }

  return (
    <div className="my-rentals-page-outdoor">
      <HeroSection />
      
      <div className="container">
        <section className="page-header-outdoor">
          <h1 className="page-title-outdoor">Мои аренды</h1>
          <p className="page-description-outdoor">
            Управляйте арендованным оборудованием, отслеживайте даты возврата и делитесь впечатлениями о походах
          </p>
        </section>

        <ImageSlider />

        <StatsSection />
        
        <PopularActivitiesSection />
        
        <TestimonialsSection />

        {rentals.length === 0 ? (
          <div className="empty-state-outdoor">
            <div className="empty-icon-wrapper-outdoor">
              <Package size={64} />
            </div>
            <h2 className="empty-title-outdoor">Нет аренд</h2>
            <p className="empty-text-outdoor">Создайте первую аренду оборудования</p>
            <button 
              className="empty-button-outdoor" 
              onClick={() => navigate('/rent')}
            >
              Арендовать оборудование
            </button>
          </div>
        ) : (
          <section className="rentals-list-outdoor">
            <div className="rentals-grid-outdoor">
              {rentals.map((rental) => (
                <RentalCard
                  key={rental.id}
                  rental={rental}
                  onReviewSubmit={handleReviewSubmit}
                  onComplete={handleComplete}
                  formatDate={formatDate}
                  getCategoryIcon={getCategoryIcon}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const RentalCard = ({ rental, onReviewSubmit, onComplete, formatDate, getCategoryIcon }) => {
  const [rating, setRating] = useState(rental.review?.rating || 0)
  const [reviewText, setReviewText] = useState('')
  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleSubmit = () => {
    if (rating === 0 || !reviewText.trim()) {
      alert('Пожалуйста, выберите рейтинг и напишите отзыв')
      return
    }
    setIsSubmitting(true)
    onReviewSubmit(rental.id, rating, reviewText)
    setIsSubmitting(false)
    setReviewText('')
    setRating(0)
  }

  const CategoryIcon = getCategoryIcon(rental.category)

  return (
    <div className={`rental-item-outdoor ${rental.status}`}>
      <div className="rental-header-outdoor">
        <div className="rental-icon-wrapper-outdoor">
          <CategoryIcon size={28} />
        </div>
        <div className="rental-info-outdoor">
          <div className="rental-number-outdoor">#{rental.id}</div>
          <div className={`status-badge-outdoor ${rental.status}`}>
            {rental.status === 'active' ? (
              <>
                <Clock size={14} />
                <span>Активна</span>
              </>
            ) : (
              <>
                <CheckCircle size={14} />
                <span>Завершена</span>
              </>
            )}
          </div>
        </div>
      </div>
      
      <h3 className="rental-title-outdoor">{rental.item}</h3>
      
      <div className="rental-dates-outdoor">
        <div className="date-item-outdoor">
          <Calendar size={16} />
          <div>
            <span className="date-label-outdoor">Начало:</span>
            <span className="date-value-outdoor">{formatDate(rental.startDate)}</span>
          </div>
        </div>
        <div className="date-item-outdoor">
          <Calendar size={16} />
          <div>
            <span className="date-label-outdoor">Возврат:</span>
            <span className="date-value-outdoor">{formatDate(rental.returnDate)}</span>
          </div>
        </div>
      </div>
      
      {rental.status === 'active' && (
        <div className="complete-section-outdoor">
          <button
            className="complete-btn-outdoor"
            onClick={() => onComplete(rental.id)}
          >
            <CheckCircle size={18} />
            <span>Завершить аренду</span>
          </button>
        </div>
      )}
      
      {rental.status === 'completed' && (
        <div className="review-section-outdoor">
          {rental.review ? (
            <div className="review-display-outdoor">
              <div className="review-stars-outdoor">
                {Array.from({ length: 5 }).map((_, i) => (
                  <Star
                    key={i}
                    size={18}
                    fill={i < rental.review.rating ? '#D97706' : 'none'}
                    color={i < rental.review.rating ? '#D97706' : '#A8A29E'}
                  />
                ))}
              </div>
              <p className="review-text-outdoor">{rental.review.text}</p>
            </div>
          ) : (
            <div className="review-form-outdoor">
              <div className="review-header-outdoor">
                <MessageSquare size={18} />
                <span>Оставить отзыв о качестве оборудования</span>
              </div>
              <p className="review-hint-outdoor">Оцените оборудование по 5-балльной шкале и поделитесь своими впечатлениями</p>
              <div className="rating-outdoor">
                <span className="rating-label-outdoor">Рейтинг:</span>
                {Array.from({ length: 5 }).map((_, i) => (
                  <button
                    key={i}
                    type="button"
                    className="star-btn-outdoor"
                    onClick={() => setRating(i + 1)}
                    title={`Оценить на ${i + 1} ${i === 0 ? 'звезду' : i < 4 ? 'звезды' : 'звезд'}`}
                  >
                    <Star
                      size={24}
                      fill={i < rating ? '#D97706' : 'none'}
                      color={i < rating ? '#D97706' : '#A8A29E'}
                    />
                  </button>
                ))}
                {rating > 0 && <span className="rating-value-outdoor">{rating} из 5</span>}
              </div>
              <div className="textarea-wrapper-outdoor">
                <label htmlFor={`review-text-${rental.id}`} className="textarea-label-outdoor">
                  Ваш отзыв:
                </label>
                <textarea
                  id={`review-text-${rental.id}`}
                  className="review-textarea-outdoor"
                  placeholder="Расскажите о вашем опыте использования оборудования..."
                  rows="4"
                  value={reviewText}
                  onChange={(e) => setReviewText(e.target.value)}
                />
              </div>
              <button
                className="submit-review-outdoor"
                onClick={handleSubmit}
                disabled={isSubmitting || rating === 0 || !reviewText.trim()}
              >
                {isSubmitting ? 'Отправка...' : 'Отправить отзыв'}
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  )
}

const StatsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)
  const [counters, setCounters] = useState({ equipment: 0, users: 0, satisfaction: 0 })

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  useEffect(() => {
    if (isVisible) {
      const duration = 2000
      const steps = 60
      const stepTime = duration / steps
      
      const animate = (key, target) => {
        let current = 0
        const increment = target / steps
        const timer = setInterval(() => {
          current += increment
          if (current >= target) {
            current = target
            clearInterval(timer)
          }
          setCounters(prev => ({ ...prev, [key]: Math.floor(current) }))
        }, stepTime)
      }

      animate('equipment', 500)
      animate('users', 1200)
      animate('satisfaction', 98)
    }
  }, [isVisible])

  const stats = [
    {
      icon: Package,
      value: `${counters.equipment}+`,
      label: 'Единиц оборудования',
      color: 'var(--outdoor-teal)'
    },
    {
      icon: TrendingUp,
      value: `${counters.users}+`,
      label: 'Довольных клиентов',
      color: 'var(--outdoor-amber)'
    },
    {
      icon: Shield,
      value: '100%',
      label: 'Гарантия качества',
      color: 'var(--outdoor-sage)'
    },
    {
      icon: BarChart3,
      value: `${counters.satisfaction}%`,
      label: 'Удовлетворенность',
      color: 'var(--outdoor-teal)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`stats-section-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="stats-header-outdoor">
        <h2 className="stats-title-outdoor">Наши достижения</h2>
        <p className="stats-description-outdoor">
          Мы гордимся качеством нашего оборудования и доверием клиентов
        </p>
      </div>
      <div className="stats-grid-outdoor">
        {stats.map((stat, index) => {
          const Icon = stat.icon
          return (
            <div 
              key={index} 
              className="stat-card-outdoor"
              style={{ 
                animationDelay: `${index * 0.1}s`,
                '--stat-color': stat.color
              }}
            >
              <div className="stat-icon-wrapper-outdoor">
                <Icon size={32} />
              </div>
              <div className="stat-value-outdoor">{stat.value}</div>
              <div className="stat-label-outdoor">{stat.label}</div>
            </div>
          )
        })}
      </div>
    </section>
  )
}

const HeroSection = () => {
  const navigate = useNavigate()
  const heroRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (heroRef.current) {
      observer.observe(heroRef.current)
    }

    return () => {
      if (heroRef.current) {
        observer.unobserve(heroRef.current)
      }
    }
  }, [])

  return (
    <section ref={heroRef} className={`hero-section-outdoor ${isVisible ? 'visible' : ''}`}>
      <div className="hero-background-outdoor"></div>
      <div className="container">
        <div className="hero-content-outdoor">
          <div className="hero-badge-outdoor">
            <Mountain size={20} />
            <span>ПРИКЛЮЧЕНИЯ НАЧИНАЮТСЯ ЗДЕСЬ</span>
          </div>
          <h1 className="hero-title-outdoor">
            Откройте мир <span className="hero-accent-outdoor">кемпинга</span>
          </h1>
          <p className="hero-description-outdoor">
            Профессиональное оборудование для ваших походов и приключений. 
            Качественное снаряжение, проверенное временем и тысячами путешественников.
          </p>
          <div className="hero-actions-outdoor">
            <button 
              className="hero-button-primary-outdoor"
              onClick={() => navigate('/rent')}
            >
              <span>Арендовать сейчас</span>
              <ArrowRight size={20} />
            </button>
            <button className="hero-button-secondary-outdoor">
              <span>Узнать больше</span>
            </button>
          </div>
          <div className="hero-stats-outdoor">
            <div className="hero-stat-outdoor">
              <div className="hero-stat-value-outdoor">500+</div>
              <div className="hero-stat-label-outdoor">Единиц оборудования</div>
            </div>
            <div className="hero-stat-outdoor">
              <div className="hero-stat-value-outdoor">1200+</div>
              <div className="hero-stat-label-outdoor">Довольных клиентов</div>
            </div>
            <div className="hero-stat-outdoor">
              <div className="hero-stat-value-outdoor">98%</div>
              <div className="hero-stat-label-outdoor">Удовлетворенность</div>
            </div>
          </div>
        </div>
      </div>
    </section>
  )
}

const PopularActivitiesSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const activities = [
    {
      icon: Mountain,
      title: 'Горные походы',
      description: 'Покорите вершины с профессиональным снаряжением',
      color: 'var(--outdoor-teal)',
      gradient: 'linear-gradient(135deg, #0F766E 0%, #14B8A6 100%)'
    },
    {
      icon: MapPin,
      title: 'Лесные маршруты',
      description: 'Исследуйте тайгу и национальные парки',
      color: 'var(--outdoor-sage)',
      gradient: 'linear-gradient(135deg, #84CC16 0%, #A3E635 100%)'
    },
    {
      icon: Camera,
      title: 'Фото-экспедиции',
      description: 'Запечатлейте красоту природы с лучшим оборудованием',
      color: 'var(--outdoor-amber)',
      gradient: 'linear-gradient(135deg, #D97706 0%, #F59E0B 100%)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`activities-section-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="activities-header-outdoor">
          <h2 className="activities-title-outdoor">Популярные активности</h2>
          <p className="activities-description-outdoor">
            Выберите свое приключение и мы подберем идеальное оборудование
          </p>
        </div>
        <div className="activities-grid-outdoor">
          {activities.map((activity, index) => {
            const Icon = activity.icon
            return (
              <div 
                key={index} 
                className="activity-card-outdoor"
                style={{ 
                  animationDelay: `${index * 0.15}s`,
                  '--activity-color': activity.color,
                  background: activity.gradient
                }}
              >
                <div className="activity-icon-wrapper-outdoor">
                  <Icon size={40} />
                </div>
                <h3 className="activity-title-outdoor">{activity.title}</h3>
                <p className="activity-text-outdoor">{activity.description}</p>
                <div className="activity-link-outdoor">
                  <span>Подробнее</span>
                  <ArrowRight size={16} />
                </div>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

const TestimonialsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const testimonials = [
    {
      name: 'Алексей Петров',
      role: 'Горный туризм',
      text: 'Отличное оборудование! Палатка выдержала сильный ветер в горах, спальник очень теплый. Всё чистое и в отличном состоянии.',
      rating: 5,
      avatar: 'AP'
    },
    {
      name: 'Мария Соколова',
      role: 'Фото-тур',
      text: 'Арендовала рюкзак и палатку для фото-экспедиции. Очень довольна качеством и сервисом. Обязательно вернусь!',
      rating: 5,
      avatar: 'МС'
    },
    {
      name: 'Дмитрий Волков',
      role: 'Лесной поход',
      text: 'Всё оборудование как новое. Особенно понравился спальник - очень теплый и удобный. Рекомендую!',
      rating: 5,
      avatar: 'ДВ'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`testimonials-section-outdoor ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="testimonials-header-outdoor">
          <h2 className="testimonials-title-outdoor">Отзывы наших клиентов</h2>
          <p className="testimonials-description-outdoor">
            Тысячи путешественников доверяют нам свои приключения
          </p>
        </div>
        <div className="testimonials-grid-outdoor">
          {testimonials.map((testimonial, index) => (
            <div 
              key={index} 
              className="testimonial-card-outdoor"
              style={{ animationDelay: `${index * 0.1}s` }}
            >
              <div className="testimonial-quote-outdoor">
                <Quote size={32} />
              </div>
              <div className="testimonial-stars-outdoor">
                {Array.from({ length: 5 }).map((_, i) => (
                  <Star
                    key={i}
                    size={18}
                    fill="#D97706"
                    color="#D97706"
                  />
                ))}
              </div>
              <p className="testimonial-text-outdoor">{testimonial.text}</p>
              <div className="testimonial-author-outdoor">
                <div className="testimonial-avatar-outdoor">
                  {testimonial.avatar}
                </div>
                <div className="testimonial-info-outdoor">
                  <div className="testimonial-name-outdoor">{testimonial.name}</div>
                  <div className="testimonial-role-outdoor">{testimonial.role}</div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

export default MyRentals

